function result = compare_columns(a,labels,style,borders,spreadCoeff)
% compare_columns(a)
% compare_columns(a,labels)
% compare_columns(a,labels,STYLE)
% compare_columns(a,labels,STYLE,BORDERS)
% compare_columns(a,labels,STYLE,BORDERS,SpreadCoeff)
%
% Give a matrix (rectangled with NaNs) or a cell array with columns,
% and it will try to compare everything with everything, 
% output the thoughts in the console, and also return the matrix of comparisons.
% If LABELS are provided, these are shown below the columns.
%
% STYLE can be any of these: 'default' (or ''),'old','paired','publication' (or 'pub'). If STYLE 
% is to be used, LABELS may be left empty [].
%
% BORDERS allows limiting axes for outliers, while still bringing outliers inside.
%
% SpreadCoeff (>1) can be used to make spreads of points more narrow. 
% Default value is 1, so try 1.5 or 2 for a start.
%
% For paired comparisons: see PAIRED_COMPARISONS

% Jan 02 2013: updated LABELS processing (if LABELS is an empty string, it is processed as if LABELS weren't passed to the function).
% May 24 2013: STYLE introduced.
% May 31 2013: Small changes in styles.
% Jan 10 2014: Reference to paired_comparisons in the help.
% Jan 16 2014: + SpreadCoeff functionality. + forced borders (not only shrinking data, but also expanding YLIM).
% Jan 24 2014: Minor consistency with updated averages_and_values funciton.
% Sep 23 2014: + test for equality of variances.
% Nov 11 2014: + new style that uses SD instead of IQR.
% Jan 14 2015: + n-values are now communicated. + rather bulky communication of ANOVA df and F-values.


ttl = '';           % Here you can put your title
plotThreshold = 15; % How far should the circles be pushed from each other in the plot. 
                    % 15 for a smaller figure; 7 for a full-screen one. May be adjusted by SPREADCOEFF arugument (it happens below).

%%% End of "manual arguments" section

% Data to compare.
% You may also provide relevant labels as a cell archive below.

if(nargin<1)
    a = [1 1 2 2 3;     0 0 0 1 2]';  % Testing purposes;
end
if(nargin<3)
    style = 'default';
end
if(nargin<4)
    borders = [];
end
if(nargin==5)
    plotThreshold = plotThreshold*spreadCoeff;
end
if(isempty(style))
    style = 'default';
end

%%% ---------------- Labels and label processing
if(exist('labels','var'))
    if(isempty(labels))
        clear('labels');
    end
end
if(~exist('labels','var'))
    %labels = {'rostral','caudal'};
    %labels = {'ybr','ybc','ypr','ypc','obr','obc','opr','opc'};
    %labels = {'yb','yp','ob','op'};
    %labels = {'Backcrash'	'Crash'	'Flash'	'Grid'	'Ramp'	'Control'};
    %labels = {'f','c','r','g','b','o'};    
    %labels = {'Flash'	'Ramp'	'Crash'	'Grid'	'Backcrash'};
    %labels = {'f','r','g','c','b'};
    for(q=1:size(a,2))
        labels{q} = num2str(q);
    end
else
    if(ischar(labels))
        temp = labels;
        labels = [];
        for(q=1:length(temp))
            labels{q} = temp(q);
        end
    end
end

%%% ---------------- Data preparation

if(~exist('labels','var'))
    for(q=1:size(a,2))
        labels{q} = num2str(q);
    end
end

if(iscell(a)) % Transform into a NaN-rectanguled matrix
    l = length(a);
    temp = 0;
    for(q=1:l)
        temp = max(temp,length(a{q}));
    end
    b = NaN(temp,l); % pre-allocation
    for(q=1:l)
        b(1:length(a{q}),q) = a{q}(:);
    end
    a = b;
    clear b;
end
   
%%% ---------------- Sanity check

if(length(labels)~=size(a,2))
    warning('Labels mismatch: I have only %d labels, but %d columns of data. Reverting to default labels.',length(labels),size(a,2));
    labels = [];
    for(q=1:size(a,2))
        labels{q} = num2str(q);
    end
end

%%% ---------------- Statistics now

[n,m] = size(a);
result = zeros(m);  % pre-allocation for the main test (MW)
varres = nan(m);    % pre-allocation for equal variance tests

fprintf('N   :');
for(q=1:m);	fprintf('\t%d',sum(~isnan(a(:,q))));    end;    fprintf('\n');
fprintf('Mean:');
for(q=1:m);	fprintf('\t%f',mean(a(~isnan(a(:,q)),q)));    end;    fprintf('\n');
fprintf('SD  :');
for(q=1:m);	fprintf('\t%f',std(a(~isnan(a(:,q)),q)));    end;    fprintf('\n');
fprintf('median:');
for(q=1:m);	fprintf('\t%f',median(a(~isnan(a(:,q)),q)));    end;    fprintf('\n');

%%% First prepare data for the ANOVA
anovaData = a(:);
anovaGroup = bsxfun(@plus,zeros(size(a)),1:m);
% ganoav = anovaGroup(:);
anovaGroup = anovaGroup(~isnan(anovaData));
anovaData = anovaData(~isnan(anovaData));
[panova,tab] = anova1(anovaData,anovaGroup,'off');
fprintf('ANOVA test: p = %s\n\n',myst(panova));
tab

fprintf('Comparison         \tMW test    \tBonferroni\n');
for(q=1:(m-1))
    for(w=(q+1):m)
%         fprintf('%d vs %d',q,w);

        %[h,result(q,w)] = kstest2(a(:,q),a(:,w)); % Kolmogorov-Smirnov
        set1 = a(:,q);
        set2 = a(:,w);
        [result(q,w),~] = ranksum(set1(~isnan(set1)),set2(~isnan(set2))); % Mann-Whitney (Wilcoxon rank sum) test
        [~,varres(q,w)] = vartest2(set1(~isnan(set1)),set2(~isnan(set2))); % Test for equal variances
        %[~,result(q,w)] = ttest2(set1(~isnan(set1)),set2(~isnan(set2))); % Mann-Whitney (Wilcoxon rank sum) test
        
        if(result(q,w)<0.05);
            fprintf('%2d vs %2d (%s-%s)   \t%s   \t%s\n',q,w,labels{q},labels{w},myst(result(q,w)),myst(result(q,w)*m*(m-1)/2));            
        end
        
%         [h,p] = ttest2(a(:,q),a(:,w));
%         if(p<0.05); fprintf(' - %5.2f Ttest',p); end
%         [h,p] = vartest2(a(:,q),a(:,w));
%         if(p<0.05); fprintf(' - %5.2f Vartest',p); end
%         fprintf('\n');
    end
end

for(q=1:(m-1))
    for(w=(q+1):m)
         if(varres(q,w)<0.05);
            fprintf('Variance test: %2d vs %2d (%s-%s)   \t%s   \t%s\n',q,w,labels{q},labels{w},myst(varres(q,w)),myst(varres(q,w)*m*(m-1)/2));            
        end
    end
end


result = result+result'+eye(m);


fprintf('\n');
formatrow = [repmat('%4f\t',1,m-1) '%4f\n'];
fprintf(formatrow,result');

%figure; myplot(-real(log(a)));

if(~isempty(borders))
    fprintf('Bringing outliers within given borders\n');
    d = borders(2)-borders(1);
    b = max(a,borders(1)+d/100);    % A safety measure to prevent these dots from disappearing from the plot
    b = min(b,borders(2)-d/100);
    b(isnan(a)) = nan;  % Restore lost NaNs
    a = b;
end

% figure; ----------------------------<<<<<<< FIGURE here
switch style
    case {'default',''}
        values_and_averages3(a,labels,plotThreshold);
    case 'old'
        values_and_averages2(a,labels,plotThreshold);
    case 'paired';
        values_and_averages2(a,labels,plotThreshold);
        hold on;
        plot(a,'g-');
        hold off;
    case {'publication','pub'}
        values_and_averages3(a,labels,plotThreshold,'pub');
    case 'mean'
        values_and_averages3(a,labels,plotThreshold,'mean');
    otherwise
        error('Unknown style');
end

if(~isempty(borders))
    ylim(borders);
end

title(ttl);

end